/****************************************************************************
 * LPC 8XX Decode program
 *
 *SCT program that implements a PWM decoder (a pulse width measurement).
 *SCT The signal frequency is assumed to be 10 kHz.
 *SCT Indicator output for missing signal.
 *SCT Indicator output for "duty cycle < min  ||  duty cycle > max"

****************************************************************************/

#include "board.h"


/* Define expected signal characteristics */
#define PWM_FREQUENCY           10000                      // PWM frequency in Hz
#define PWM_RESOLUTION_NS       1000                       // Timer resolution in ns
#define PWM_MIN_DUTY_PERCENT    25                         // Minimum allowed duty cycle in %
#define PWM_MAX_DUTY_PERCENT    70                         // Maximum allowed duty cycle in %

/* Derived constants */
#define SCT_PRESCALER           (((SystemCoreClock / 1000u) * PWM_RESOLUTION_NS) / 1000000u - 1u)

#define match_min_width         ((10000000u * PWM_MIN_DUTY_PERCENT) / (PWM_FREQUENCY * PWM_RESOLUTION_NS))
#define match_max_width         ((10000000u * PWM_MAX_DUTY_PERCENT) / (PWM_FREQUENCY * PWM_RESOLUTION_NS))
#define match_no_input          ((10000000u * 300                 ) / (PWM_FREQUENCY * PWM_RESOLUTION_NS))


void sct_fsm_init (void);


void sct_fsm_init(void)
{

	Chip_SCT_Init(LPC_SCT);							   					                                    //enable clock for SCT0/1

	Chip_SCT_Config(LPC_SCT,  SCT_CONFIG_32BIT_COUNTER | SCT_CONFIG_AUTOLIMIT_L); 				//Unified

	Chip_SCT_SetControl(LPC_SCT, SCT_CTRL_PRE_L(SCT_PRESCALER));		                    // prescaler


	LPC_SCT->REGMODE_L      = 0x00000018;                 				// 3x MATCH, 2x CAPTURE used

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_0, match_max_width);		// match_max_width
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_0, match_max_width);

	Chip_SCT_SetMatchCount(LPC_SCT,SCT_MATCH_1,match_min_width);		// match_min_width
	Chip_SCT_SetMatchReload(LPC_SCT,SCT_MATCH_1,match_min_width );

	Chip_SCT_SetMatchCount(LPC_SCT,SCT_MATCH_2,match_no_input);			// match_no_input
	Chip_SCT_SetMatchReload(LPC_SCT,SCT_MATCH_2,match_no_input );


	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_0, ENABLE_ALL_STATES);		// event 0 happens in all states
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_0, (CHIP_SCT_EVENTCTRL_T) (  SCT_EVECTRL_MATCH2 |
                                                                        SCT_COMBMODE_MATCH ));	// related to match_no_input only

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_1, ENABLE_ALL_STATES);		// event 1 happens in all states
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_1, (CHIP_SCT_EVENTCTRL_T) (  SCT_IOCOND_RAISE |
                                                                        SCT_COMBMODE_IO  )); 	  // IN_0 rising edge condition only

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_2, ENABLE_STATE0);			// event 2 happens in state 0
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_2, (CHIP_SCT_EVENTCTRL_T) (  SCT_EVECTRL_MATCH1  | 	// related to match_min_width
                                                                        SCT_COMBMODE_MATCH  |	  // match condition only
                                                                        SCT_STATELD_1		    | 	// STATEV is loaded into state
                                                                        SCT_STATEEV_1		    ));	// new state is 1

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_3, ENABLE_STATE1);		    // event 3 happens in state 1
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_3, (CHIP_SCT_EVENTCTRL_T) (  SCT_IOCOND_FALL  	| 	// IN_0 falling edge
                                                                        SCT_COMBMODE_IO   |	  // IO condition only
                                                                        SCT_STATELD_1		  | 	// STATEV is loaded into state
                                                                        SCT_STATEEV_0		  ));	// new state is 0

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_4, ENABLE_STATE1);		    // event 4 happens in state 1
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_4, (CHIP_SCT_EVENTCTRL_T) (  SCT_EVECTRL_MATCH0  | 	// related to match_max_width
                                                                        SCT_COMBMODE_MATCH  |	  // match condition only
                                                                        SCT_STATELD_1		    | 	// STATEV is loaded into state
                                                                        SCT_STATEEV_0		    ));	// new state is 0


	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_5, ENABLE_STATE0);		    // event 5 happens in state 0
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_5, (CHIP_SCT_EVENTCTRL_T) (  SCT_IOCOND_FALL  	  | 	// IN_0 falling edge
                                                                        SCT_COMBMODE_IO     ));	// new state is 0


	Chip_SCT_CapControl(LPC_SCT, SCT_CAPCTRL_3 , SCT_EVT_1);			                  // event 1 is causing capture 3
	Chip_SCT_CapControl(LPC_SCT, SCT_CAPCTRL_4 , (CHIP_SCT_EVENT_T) ( SCT_EVT_3 |
                                                                    SCT_EVT_5 ));	// event 3 and 5 cause capture 4


	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_0, SCT_EVT_1);				                    // event 1 set   OUT0 (no timeout)
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_0, SCT_EVT_0);				                  // event 0 clear OUT0 (timeout)

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_1, SCT_EVT_3);				                    // event 3 set   OUT1 (no width error)
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_1, (CHIP_SCT_EVENT_T)  ( SCT_EVT_0 |
                                                                    SCT_EVT_5 ));	// event 0 and 5 clear OUT1 (width error)

	Chip_SCT_Output(LPC_SCT, 0x03);        								                          // default set OUT0 and OUT1

	LPC_SCT->LIMIT_L        = (1 << 0) | (1 << 1);        				                  // event 0 and 1 limit the timer

	Chip_SCT_EnableEventInt(LPC_SCT,	(CHIP_SCT_EVENT_T)  ( SCT_EVT_0 |
                                                          SCT_EVT_5 ));	   				// event 0 and 5 generate an irq


	NVIC_EnableIRQ(SCT_IRQn);                             				                  // enable SCT interrupt

	Chip_SCT_ClearControl(LPC_SCT,SCT_CTRL_HALT_L);	   					                    // unhalt it by clearing bit 2 of the CTRL register

}


void SCT_IRQHandler(void)
{
	uint32_t status = Chip_SCT_GetEventFlag(LPC_SCT);

	if (status & SCT_EVE_0)								 // EVENT 0: timeout
	{
			/* New measurement result */
			;
	}
	if (status & SCT_EVE_5)
	{
			/* Time-out (no signal) */
			;
	}

	LPC_SCT->EVFLAG = status;			// EVENT 5: width error
}



int main (void)
{

  // Force the counter to be placed into memory
	volatile static int i = 0 ;
  
	SystemCoreClockUpdate();
	Board_Init();


	Chip_Clock_EnablePeriphClock(SYSCTL_CLOCK_IOCON);								// enable clock for IOCON

	Chip_SWM_Init();
  
  Chip_SWM_DisableFixedPin(SWM_FIXED_ACMP_I1);                    //disable Pinmux for P0_0 (ACMP_I1)

	Chip_SWM_MovablePinAssign(SWM_CTIN_0_I ,0);									    // SCT_IN0  at P0_0 = PWM signal input
	Chip_SWM_MovablePinAssign(SWM_CTOUT_0_O, 7);									  // SCT_OUT0 at P0_7 = timeout (red LED)
	Chip_SWM_MovablePinAssign(SWM_CTOUT_1_O , 17);									// SCT_OUT1 at P0_17 = width_error (green LED)


	sct_fsm_init();                                   								// Initialize SCT
	
	// Enter an infinite loop
	while(1)
	{
		/* The settings in the_real_sct_user.h have been chosen such that i
		 * is set in % duty cycle (assuming a PWM frequency of 10 kHz).
		 * (In a real world application you want to select better resolution!)
		 */
		i = LPC_SCT->CAP[4].U;

		__WFI();
	}
}
